(function () {
  'use strict';

  const navToggle = document.querySelector('[data-nav-toggle]');
  const navPanel = document.querySelector('[data-nav-panel]');

  if (navToggle && navPanel) {
    navToggle.addEventListener('click', () => {
      const isOpen = navPanel.classList.toggle('is-open');
      navToggle.setAttribute('aria-expanded', String(isOpen));
    });

    document.addEventListener('click', (e) => {
      if (!navPanel.classList.contains('is-open')) return;
      const within = navPanel.contains(e.target) || navToggle.contains(e.target);
      if (!within) {
        navPanel.classList.remove('is-open');
        navToggle.setAttribute('aria-expanded', 'false');
      }
    });
  }

  // Submenu toggle buttons (mobile)
  document.querySelectorAll('.nav__item--has-children').forEach((item) => {
    const btn = item.querySelector('.nav__toggle');
    const submenu = item.querySelector('.nav__submenu');
    if (!btn || !submenu) return;

    btn.addEventListener('click', (e) => {
      e.preventDefault();
      const open = item.classList.toggle('is-open');
      btn.setAttribute('aria-expanded', String(open));
    });
  });

  // Quote form mobile pill toggle
  document.querySelectorAll('[data-quote-toggle]').forEach((btn) => {
    btn.addEventListener('click', () => {
      const wrap = btn.closest('[data-quote]');
      if (!wrap) return;
      const open = wrap.classList.toggle('is-open');
      btn.setAttribute('aria-expanded', String(open));
    });
  });

  // Scroll reveal
  const revealEls = Array.from(document.querySelectorAll('.reveal'));
  if ('IntersectionObserver' in window && revealEls.length) {
    const io = new IntersectionObserver((entries) => {
      entries.forEach((entry) => {
        if (entry.isIntersecting) {
          entry.target.classList.add('is-in');
          io.unobserve(entry.target);
        }
      });
    }, { threshold: 0.15 });

    revealEls.forEach((el) => io.observe(el));
  } else {
    revealEls.forEach((el) => el.classList.add('is-in'));
  }

  // Gentle inline validation (shows messages only after submit attempt)
  function setFieldError(field, message) {
    const wrap = field.closest('.quote__field') || field.closest('.field') || field.parentElement;
    if (!wrap) return;
    wrap.classList.add('is-error');

    let msg = wrap.querySelector('.field-error');
    if (!msg) {
      msg = document.createElement('div');
      msg.className = 'field-error';
      msg.setAttribute('role', 'alert');
      wrap.appendChild(msg);
    }
    msg.textContent = message;
    field.setAttribute('aria-invalid', 'true');
  }

  function clearFieldError(field) {
    const wrap = field.closest('.quote__field') || field.closest('.field') || field.parentElement;
    if (!wrap) return;
    wrap.classList.remove('is-error');
    const msg = wrap.querySelector('.field-error');
    if (msg) msg.remove();
    field.removeAttribute('aria-invalid');
  }

  function validateForm(form) {
    let ok = true;
    const required = Array.from(form.querySelectorAll('input[required], textarea[required]'));
    required.forEach((field) => {
      clearFieldError(field);
      const val = (field.value || '').trim();

      if (!val) {
        ok = false;
        setFieldError(field, 'Required');
        return;
      }
      if (field.type === 'email') {
        const valid = /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(val);
        if (!valid) {
          ok = false;
          setFieldError(field, 'Enter a valid email');
        }
      }
    });
    return ok;
  }

  document.querySelectorAll('form[data-validate]').forEach((form) => {
    form.addEventListener('submit', (e) => {
      const ok = validateForm(form);
      if (!ok) {
        e.preventDefault();
        const first = form.querySelector('[aria-invalid="true"]');
        if (first) first.focus();
      }
    });

    form.querySelectorAll('input, textarea').forEach((field) => {
      field.addEventListener('input', () => clearFieldError(field));
    });
  });

  // Results filters (portfolio)
  const filterButtons = Array.from(document.querySelectorAll('[data-filter]'));
  const grid = document.querySelector('[data-portfolio-grid]');
  if (filterButtons.length && grid) {
    filterButtons.forEach((btn) => {
      btn.addEventListener('click', () => {
        filterButtons.forEach((b) => b.classList.remove('is-active'));
        btn.classList.add('is-active');
        const filter = btn.getAttribute('data-filter');

        Array.from(grid.querySelectorAll('.portfolio-card')).forEach((card) => {
          const type = card.getAttribute('data-type');
          const show = (filter === 'all') || (type === filter);
          card.classList.toggle('is-hidden', !show);
        });
      });
    });
  }
})();
